<?php
/* =====================================================
   R2 WORKER FOR SUB-ACCOUNTS (LOAD BALANCER NODE)
   Path: /public_html/new/cron/r2-hls-worker.php
   NOTE: Points to Main Account Data (/shop1bollydrive)
===================================================== */

set_time_limit(0);
ini_set('memory_limit','-1');
error_reporting(E_ALL);

// 🔥🔥 MAGIC LINE: Sabhi Sub-Accounts Main Account ka Data use karenge 🔥🔥
$BASE    = '/var/www/shop1bollydrive/public_html/new';

// Baki paths automatic $BASE se ban jayenge
$SHARE   = "$BASE/base/data/main/share";
$STORAGE = "$BASE/storage"; 
$TMP     = "$BASE/storage/tmp";
$HLS     = "$BASE/storage/hls";

// Tools Paths (Check karein ki ye sub-accounts par bhi yahi hain)
$RCLONE      = '/usr/bin/rclone';
$FFMPEG      = '/usr/bin/ffmpeg';
$FFPROBE     = '/usr/bin/ffprobe';

// Config bhi Main Account ki use hogi
$RCLONE_CONF = "$BASE/storage/rclone.conf";
$R2_DEST     = "bollydrive:bollydrive"; 
$FORCE_REMOTE_NAME = "gdrive"; 

// --- HELPER FUNCTIONS ---
function hardDelete($path){
    if (!file_exists($path)) return true;
    if (is_file($path) || is_link($path)) return @unlink($path);
    $files = scandir($path);
    foreach ($files as $file) {
        if ($file === '.' || $file === '..') continue;
        hardDelete($path . '/' . $file);
    }
    return @rmdir($path);
}

function runCommandWithRetry($cmd, $description) {
    $maxRetries = 2; $attempt = 0;
    while ($attempt < $maxRetries) {
        $attempt++; $output = []; $returnCode = 0;
        exec("$cmd 2>&1", $output, $returnCode);
        if ($returnCode === 0) return true;
        sleep(2);
    }
    throw new Exception("FAILED: $description.\n" . implode("\n", $output));
}

function getDriveAccessToken($rcloneBin, $confPath, $remoteName) {
    $cmdRefresh = "$rcloneBin about ".escapeshellarg("$remoteName:")." --config ".escapeshellarg($confPath)." 2>&1";
    exec($cmdRefresh, $refOut, $refCode);
    $cmdDump = "$rcloneBin config dump --config ".escapeshellarg($confPath);
    $jsonOutput = shell_exec($cmdDump);
    if (!$jsonOutput) throw new Exception("Rclone config dump failed.");
    $configData = json_decode($jsonOutput, true);
    if (isset($configData[$remoteName]['token'])) {
        $tokenData = json_decode($configData[$remoteName]['token'], true);
        if (isset($tokenData['access_token'])) return $tokenData['access_token'];
    }
    throw new Exception("Remote '$remoteName' token not found.");
}

try {
    // 1. SETUP JOB
    if (!file_exists($RCLONE_CONF)) exit("❌ CRITICAL: Config missing at $RCLONE_CONF\n");
    
    // 🔥 Worker ko lock lagana padega taaki 2 workers same file na utha lein
    $jobFile = null; $jobId = null; $jobData = null;
    
    // Scan shared folder
    foreach (glob("$SHARE/*.json") as $f) {
        $d = json_decode(file_get_contents($f), true);
        
        // Sirf wahi uthao jo 'queued' hai aur abhi start nahi hua
        if (($d['r2_status'] ?? '') === 'queued' && empty($d['r2_started'])) {
            
            // Race Condition Fix: Turant file lock karke check karo
            $fp = fopen($f, 'c+');
            if (flock($fp, LOCK_EX | LOCK_NB)) { // Non-blocking lock
                $d = json_decode(file_get_contents($f), true); // Re-read inside lock
                if (($d['r2_status'] ?? '') === 'queued' && empty($d['r2_started'])) {
                    $jobFile = $f; $jobId = basename($f, '.json'); $jobData = $d;
                    
                    // Mark as started immediately inside lock
                    $jobData['r2_status'] = 'uploading';
                    $jobData['r2_started'] = time();
                    ftruncate($fp, 0); rewind($fp);
                    fwrite($fp, json_encode($jobData, JSON_PRETTY_PRINT));
                    flock($fp, LOCK_UN); fclose($fp);
                    break; // Job mil gaya!
                }
                flock($fp, LOCK_UN);
            }
            fclose($fp);
        }
    }
    
    if (!$jobId) exit("No queued job found.\n");

    // 2. DOWNLOAD
    $targetFileId = $jobData['file']['file_id'] ?? '';
    if (!$targetFileId) throw new Exception("File ID missing");
    
    $found = null;
    foreach (glob("$STORAGE/drive_index_*.json") as $iFile) {
        $chunkData = json_decode(file_get_contents($iFile), true);
        if (!$chunkData) continue;
        foreach ($chunkData as $f) {
            if (($f['id'] ?? '') === $targetFileId) { $found = $f; break 2; }
        }
    }
    if (!$found) throw new Exception("File ID not found in index.");

    $remote = (isset($FORCE_REMOTE_NAME) && !empty($FORCE_REMOTE_NAME)) ? $FORCE_REMOTE_NAME : $found['remote'];
    
    // Temp folder (Is baar s1, s2 apne local folder mein download karenge, main mein nahi)
    // Taki load distribute ho
    $LOCAL_TMP = __DIR__ . "/../storage/tmp"; 
    @mkdir("$LOCAL_TMP/$jobId", 0777, true); 
    
    // HLS folder bhi local banega pehle
    $LOCAL_HLS = __DIR__ . "/../storage/hls";
    @mkdir("$LOCAL_HLS/$jobId", 0777, true);

    $localFile = "$LOCAL_TMP/$jobId/video_source.mkv";

    echo "Downloading ID: {$found['id']} on Sub-Node...\n";
    
    $accessToken = getDriveAccessToken($RCLONE, $RCLONE_CONF, $remote);
    if (!$accessToken) throw new Exception("Could not fetch Access Token.");

    $fp = fopen($localFile, 'w+');
    $ch = curl_init("https://www.googleapis.com/drive/v3/files/{$found['id']}?alt=media&acknowledgeAbuse=true");
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Authorization: Bearer $accessToken"]);
    curl_setopt($ch, CURLOPT_FILE, $fp);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 0);
    curl_setopt($ch, CURLOPT_FAILONERROR, true);
    if (!curl_exec($ch)) { 
        $err = curl_error($ch); fclose($fp); 
        throw new Exception("Download Failed: $err"); 
    }
    curl_close($ch); fclose($fp);

    if (!file_exists($localFile) || filesize($localFile) < 1024) throw new Exception("Empty File.");
    echo "✅ Download Success.\n";

    // 3. PROCESSING
    $cmdProbe = "$FFPROBE -v error -show_entries stream=index,codec_type:stream_tags=language,title -select_streams a -of json ".escapeshellarg($localFile);
    $probeData = json_decode(shell_exec($cmdProbe), true);
    $audioTracks = [];
    if (!empty($probeData['streams'])) {
        foreach ($probeData['streams'] as $s) $audioTracks[] = ['index'=>$s['index'],'lang'=>$s['tags']['language']??'und','title'=>$s['tags']['title']??''];
    }
    
    $knownLanguages = ['Hindi', 'English', 'Tamil', 'Telugu', 'Malayalam', 'Kannada', 'Marathi', 'Punjabi', 'Bengali', 'Urdu', 'Japanese', 'Korean'];
    $defaultAudioIndex = 0;
    foreach ($audioTracks as $k => $track) {
        if (stripos($track['lang'], 'hin') !== false || stripos($track['title'], 'hindi') !== false) { $defaultAudioIndex = $k; break; }
    }

    $audioManifestLines = [];
    $audioGroupId = "stereo";

    foreach ($audioTracks as $k => $track) {
        $cleanName = "";
        foreach ($knownLanguages as $lang) { if (stripos($track['title'], $lang) !== false) { $cleanName = $lang; break; } }
        if (!$cleanName) {
            if (stripos($track['lang'], 'hin') !== false) $cleanName = "Hindi";
            elseif (stripos($track['lang'], 'eng') !== false) $cleanName = "English";
            elseif (stripos($track['lang'], 'tam') !== false) $cleanName = "Tamil";
            elseif (stripos($track['lang'], 'tel') !== false) $cleanName = "Telugu";
        }
        if (!$cleanName) {
            $junk = ['vegamovies', 'mkvcinemas', 'bollydrive', '.is', '.com', '.net', '.mkv', '[', ']', '(', ')', '-'];
            $cleanName = ucwords(trim(preg_replace('/\s+/', ' ', str_replace($junk, ' ', strtolower($track['title'])))));
        }
        if (empty($cleanName) || strlen($cleanName) < 2) $cleanName = "Audio " . ($k + 1);

        $isDefault = ($k === $defaultAudioIndex) ? "YES" : "NO";
        $aDir = "$LOCAL_HLS/$jobId/hls/audio_{$k}"; @mkdir($aDir, 0777, true);

        runCommandWithRetry("$FFMPEG -y -i ".escapeshellarg($localFile)." -map 0:".$track['index']." -c:a aac -b:a 128k -ac 2 -f hls -hls_time 10 -hls_list_size 0 -hls_flags independent_segments -hls_segment_filename ".escapeshellarg("$aDir/seg_%05d.ts")." ".escapeshellarg("$aDir/index.m3u8"), "Audio Encode");
        
        if (file_exists("$aDir/index.m3u8")) {
            runCommandWithRetry("$RCLONE copy ".escapeshellarg($aDir)." ".escapeshellarg("$R2_DEST/$jobId/hls/audio_{$k}")." --config ".escapeshellarg($RCLONE_CONF), "Upload Audio");
            $audioManifestLines[] = "#EXT-X-MEDIA:TYPE=AUDIO,GROUP-ID=\"$audioGroupId\",LANGUAGE=\"{$track['lang']}\",NAME=\"{$cleanName}\",DEFAULT=$isDefault,AUTOSELECT=$isDefault,URI=\"audio_{$k}/index.m3u8\"";
        }
    }

    $w = intval(shell_exec("$FFPROBE -v error -select_streams v:0 -show_entries stream=width -of csv=p=0 ".escapeshellarg($localFile)));
    if ($w < 1280) throw new Exception("Source < 720p ($w)");

    $profiles = ['480'=>['width'=>854,'vb'=>'1200k'], '720'=>['width'=>1280,'vb'=>'2500k'], '1080'=>['width'=>1920,'vb'=>'4500k']];
    $uploadedQ = []; $vidLines = [];

    foreach ($profiles as $q => $p) {
        if ($w < ($p['width'] - 10)) continue;
        $outDir = "$LOCAL_HLS/$jobId/hls/$q"; @mkdir($outDir, 0777, true);
        try {
            runCommandWithRetry("$FFMPEG -y -i ".escapeshellarg($localFile)." -vf scale=".$p['width'].":-2 -c:v libx264 -preset ultrafast -crf 26 -tune fastdecode -threads 0 -maxrate ".$p['vb']." -bufsize ".$p['vb']." -g 48 -sc_threshold 0 -an -f hls -hls_time 8 -hls_list_size 0 -hls_flags independent_segments -hls_segment_filename ".escapeshellarg("$outDir/seg_%05d.ts")." ".escapeshellarg("$outDir/index.m3u8"), "Encode $q");
            if (!file_exists("$outDir/index.m3u8")) throw new Exception("Playlist missing");
            runCommandWithRetry("$RCLONE copy ".escapeshellarg($outDir)." ".escapeshellarg("$R2_DEST/$jobId/hls/$q")." --config ".escapeshellarg($RCLONE_CONF), "Upload $q");
            $uploadedQ[] = $q;
            $bw = (intval($p['vb']) * 1000) + 128000;
            $vidLines[] = "#EXT-X-STREAM-INF:BANDWIDTH=$bw,RESOLUTION=".$p['width']."x0,AUDIO=\"stereo\"\n$q/index.m3u8";
        } catch (Exception $e) { if($q=='480'||$q=='720') throw $e; }
    }

    if (!in_array('480', $uploadedQ) || !in_array('720', $uploadedQ)) throw new Exception("Mandatory Quality Missing");

    $m3u8 = "#EXTM3U\n#EXT-X-VERSION:3\n\n" . implode("\n", $audioManifestLines) . "\n\n" . implode("\n", $vidLines) . "\n";
    file_put_contents("$LOCAL_HLS/$jobId/hls/index.m3u8", $m3u8);
    runCommandWithRetry("$RCLONE copy ".escapeshellarg("$LOCAL_HLS/$jobId/hls/index.m3u8")." ".escapeshellarg("$R2_DEST/$jobId/hls/")." --config ".escapeshellarg($RCLONE_CONF), "Master Upload");

    // FINALIZE
    $jobData['r2_status'] = 'ready';
    $jobData['r2_ready_at'] = time();
    unset($jobData['r2_started']);
    
    // 🔥 Final Status Update Main Account par
    $fp = fopen($jobFile, 'c+');
    if (flock($fp, LOCK_EX)) {
        ftruncate($fp, 0); rewind($fp);
        fwrite($fp, json_encode($jobData, JSON_PRETTY_PRINT));
        flock($fp, LOCK_UN);
    }
    fclose($fp);
    echo "JOB DONE: $jobId\n";

} catch (Throwable $e) {
    if ($jobFile) {
        $d = json_decode(file_get_contents($jobFile), true);
        $d['r2_status'] = 'error';
        $d['r2_error'] = substr($e->getMessage(),0,800);
        unset($d['r2_started']);
        file_put_contents($jobFile, json_encode($d, JSON_PRETTY_PRINT));
    }
    echo "ERROR: " . $e->getMessage() . "\n";
    if ($jobId) exec("$RCLONE purge ".escapeshellarg("$R2_DEST/$jobId")." --config ".escapeshellarg($RCLONE_CONF));
} finally {
    // Local cleanup on Sub-Account
    if(isset($LOCAL_TMP) && $jobId) hardDelete("$LOCAL_TMP/$jobId");
    if(isset($LOCAL_HLS) && $jobId) hardDelete("$LOCAL_HLS/$jobId");
}
?>